/*
 * File: DateSlider.java
 * ==================================================================
 * A custom slider control that slides over a range of dates.  The
 * control contains two sliders that let you define a range of dates.
 */
import java.awt.*;
import java.awt.event.*;
import java.text.*;
import java.util.*;
import javax.swing.*;
import javax.swing.event.*;

public class DateSlider extends JPanel implements ChangeListener {
	/* The constant by which to scale up/down the date. */
	private static final long SCALING_FACTOR = 100000;
	
	/* Our listeners. */
	private Set<ActionListener> listeners = new HashSet<ActionListener>();

	/* Labels displaying the current dates. */
	private JLabel startLabel, endLabel;
	
	/* The actual slider bars. */
	private JSlider startSlider, endSlider;
	
	/**
	 * Constructs a DateSlider over the given range of dates.
	 * 
	 * @param startDate The starting date.
	 * @param endDate The ending date.
	 */
	public DateSlider(Date startDate, Date endDate) {
		/* Dates are internally represented as longs holding the number of
		 * milliseconds from January 1, 1970.  Since this value is way too
		 * big to store in an int, we divide everything by a scaling factor
		 * so that the slider bar can actually work.
		 */
		int startMillis = (int)(startDate.getTime() / SCALING_FACTOR);
		int endMillis = (int)(endDate.getTime() / SCALING_FACTOR);
		
		/* Use a grid layout to stack the components into rows. */
		setLayout(new GridLayout(2, 2));
		
		/* Create the date and slider fields. */
		startLabel = new JLabel();
		endLabel = new JLabel();
		startSlider = new JSlider(startMillis, endMillis, startMillis);
		endSlider = new JSlider(startMillis, endMillis, startMillis);
		
		/* Respond to changes in the sliders. */
		startSlider.addChangeListener(this);
		endSlider.addChangeListener(this);
		
		/* Compute the text to label the sliders. */
		updateText();
		
		add(startLabel);
		add(endLabel);
		add(startSlider);
		add(endSlider);
		
		/* Ask that we be a bit bigger than usual. */
		Dimension size = getPreferredSize();
		size.setSize(2 * size.getWidth(), size.getHeight());
		setPreferredSize(size);
	}
	
	/**
	 * Adds a listener for this slider that will be notified whenever an action
	 * is performed.
	 * 
	 * @param listener The listener to add.
	 */
	public void addActionListener(ActionListener listener) {
		listeners.add(listener);
	}
	
	/**
	 * Retrieves the start Date encoded by the slider bar.
	 * 
	 * @return The start date encoded by the slider bar.
	 */
	public Date getStartDate() {
		/* Convert from the slider position to a calendar date. */
		Calendar c = new GregorianCalendar();
		c.setTimeInMillis(startSlider.getValue() * SCALING_FACTOR);
		
		/* Convert this into a Date object. */
		return c.getTime();
	}
	
	/**
	 * Retrieves the end Date encoded by the slider bar.
	 * 
	 * @return The end date encoded by the slider bar.
	 */
	public Date getEndDate() {
		/* Convert from the slider position to a calendar date. */
		Calendar c = new GregorianCalendar();
		c.setTimeInMillis(endSlider.getValue() * SCALING_FACTOR);
		
		/* Convert this into a Date object. */
		return c.getTime();
	}
	
	/**
	 * Updates the description of what day the slider is at.
	 */
	private void updateText() {		
		DateFormat df = new SimpleDateFormat("MMMM, yyyy");
		startLabel.setText(df.format(getStartDate().getTime()));
		endLabel.setText(df.format(getEndDate().getTime()));
	}
	
	/**
	 * Responds to the slider bar changing.
	 */
	public void stateChanged(ChangeEvent e) {
		/* If a slider bar has changed to define an invalid range, change the
		 * other slider bar.
		 */
		if (e.getSource() == startSlider && getEndDate().before(getStartDate())) {
			endSlider.setValue((int)(getStartDate().getTime() / SCALING_FACTOR));
		}
		if (e.getSource() == endSlider && getEndDate().before(getStartDate())) {
			startSlider.setValue((int)(getEndDate().getTime() / SCALING_FACTOR));
		}
		
		/* Redraw the text on the sliders. */
		updateText();
		
		/* Tell all of our listeners that something happened. */
		for (ActionListener listener: listeners) {
			listener.actionPerformed(new ActionEvent(this, ActionEvent.ACTION_FIRST, ""));
		}
	}
}
